﻿#include "precompiled.h"
#include "common.h"
#include "Scene.h"

#include "Camera.h"
#include "Entity.h"
#include "Transform.h"
#include "MeshRenderer.h"
#include "ResourceManager.h"

using namespace DirectX;

namespace RTCam {

Scene::Scene(void) :
	m_activeCam(),
	m_resources()
{
}


Scene::~Scene(void)
{
}

shared_ptr<Entity> Scene::AddCamera()
{
	// Create the object
	shared_ptr<Entity> camera = Entity::Create("Camera");
	m_entities.push_back(camera);

	// Initialize the component
	camera->AddCamera(m_resources.lock());

	// If there's no active camera in the scene yet, make this the active camera.
	if(m_activeCam.expired()) {
		m_activeCam = camera->GetCamera();
	}
	
	return camera;
}

shared_ptr<Entity> Scene::AddCube(const XMFLOAT3A& position)
{
	ASSERT_MSG(m_resources.lock() != nullptr, "A scene did not have access to the resource manager to create an entity.");

	// Create the object
	shared_ptr<Entity> cube = Entity::Create("Cube");
	m_entities.push_back(cube);

	// Set position
	cube->GetTransform()->SetLocalPosition(position);

	// Add a cube mesh
	cube->AddMeshRenderer(m_resources.lock()->m_meshCube);

	return cube;
}

shared_ptr<Entity> Scene::AddColorCube(const XMFLOAT3A& position)
{
	ASSERT_MSG(m_resources.lock() != nullptr, "A scene did not have access to the resource manager to create an entity.");

	// Create the object
	shared_ptr<Entity> cube = Entity::Create("ColorCube");
	m_entities.push_back(cube);

	// Set position
	cube->GetTransform()->SetLocalPosition(position);

	// Add a cube mesh
	cube->AddMeshRenderer(m_resources.lock()->m_meshColorCube);

	return cube;
}

void Scene::Initialize( const shared_ptr<class ResourceManager>& resources )
{
	m_resources = resources;
}

void Scene::AddEntity(const shared_ptr<Entity>& entity)
{
	// Make sure the entity isn't already in the scene
	for(auto& e : m_entities) {
		if(e.get() == entity.get()) {
			DebugPrint("WARNING: Attempted to add entity %s more than once into a scene.\n", entity->m_name.c_str());
			return;
		}
	}

	m_entities.push_back(entity);
}

void Scene::RemoveEntity( const shared_ptr<Entity>& entity)
{
	if(m_activeCam.lock().get() == entity->GetCamera().get()) {
		m_activeCam.reset();
	}

	for(auto curIter = m_entities.begin(); curIter != m_entities.end(); ++curIter) {
		if(curIter->get() == entity.get()) {
			m_entities.erase(curIter);
			return;
		}
	}
}

} // end namespace